// Sistema de Controle de Loop do Funil
(function() {
    'use strict';
    
    const LoopControl = {
        // Configurações do sistema de loop
        config: {
            maxLoops: 6, // Máximo de 6 loops completos
            sessionKey: 'funnelLoopCount',
            pagesKey: 'funnelPagesVisited',
            startTimeKey: 'funnelStartTime',
            cooldownTime: 30 * 60 * 1000, // 30 minutos de cooldown
            exitUrl: 'https://go.invictuspay.app.br/xyh7k9w1fe'
        },
        
        // Inicializa o controle de loop
        init() {
            this.trackPageVisit();
            this.checkLoopStatus();
            this.setupExitHandlers();
        },
        
        // Rastreia a visita da página atual
        trackPageVisit() {
            const currentPage = this.getCurrentPage();
            const visitedPages = this.getVisitedPages();
            const startTime = this.getStartTime();
            
            // Se é a primeira visita, inicializa o tracking
            if (!startTime) {
                sessionStorage.setItem(this.config.startTimeKey, Date.now().toString());
            }
            
            // Adiciona a página atual à lista de visitadas
            if (!visitedPages.includes(currentPage)) {
                visitedPages.push(currentPage);
                sessionStorage.setItem(this.config.pagesKey, JSON.stringify(visitedPages));
            }
            
            // Se voltou para a página inicial, incrementa o contador de loops
            if (currentPage === 'index' && visitedPages.length > 1) {
                this.incrementLoopCount();
            }
        },
        
        // Verifica o status do loop e toma ações se necessário
        checkLoopStatus() {
            const loopCount = this.getLoopCount();
            const startTime = this.getStartTime();
            const currentTime = Date.now();
            
            // Verifica se passou do tempo de cooldown
            if (startTime && (currentTime - startTime) > this.config.cooldownTime) {
                this.resetLoopData();
                return;
            }
            
            // Se excedeu o número máximo de loops
            if (loopCount >= this.config.maxLoops) {
                this.handleMaxLoopsReached();
                return;
            }
            
            // Aviso quando está próximo do limite
            if (loopCount === this.config.maxLoops - 1) {
                this.showLoopWarning();
            }
        },
        
        // Obtém a página atual baseada na URL
        getCurrentPage() {
            const path = window.location.pathname;
            
            if (path.includes('taxa') && !path.includes('taxadevolta')) {
                return 'taxa';
            } else if (path.includes('iof')) {
                return 'iof';
            } else if (path.includes('taxadevolta')) {
                return 'taxadevolta';
            } else if (path.includes('sucesso')) {
                return 'sucesso';
            } else {
                return 'index';
            }
        },
        
        // Obtém o contador de loops atual
        getLoopCount() {
            return parseInt(sessionStorage.getItem(this.config.sessionKey) || '0');
        },
        
        // Incrementa o contador de loops
        incrementLoopCount() {
            const currentCount = this.getLoopCount();
            sessionStorage.setItem(this.config.sessionKey, (currentCount + 1).toString());
        },
        
        // Obtém as páginas visitadas
        getVisitedPages() {
            const stored = sessionStorage.getItem(this.config.pagesKey);
            return stored ? JSON.parse(stored) : [];
        },
        
        // Obtém o tempo de início da sessão
        getStartTime() {
            const stored = sessionStorage.getItem(this.config.startTimeKey);
            return stored ? parseInt(stored) : null;
        },
        
        // Reseta os dados de loop
        resetLoopData() {
            sessionStorage.removeItem(this.config.sessionKey);
            sessionStorage.removeItem(this.config.pagesKey);
            sessionStorage.removeItem(this.config.startTimeKey);
        },
        
        // Manipula quando o máximo de loops é atingido
        handleMaxLoopsReached() {
            
            // Mostra uma mensagem para o usuário
            if (typeof Swal !== 'undefined') {
                Swal.fire({
                    title: 'Limite Atingido',
                    text: 'Você já passou pelo processo várias vezes. Redirecionando para finalizar...',
                    icon: 'info',
                    timer: 3000,
                    showConfirmButton: false
                }).then(() => {
                    this.redirectToExit();
                });
            } else {
                alert('Você já passou pelo processo várias vezes. Redirecionando...');
                setTimeout(() => this.redirectToExit(), 2000);
            }
        },
        
        // Mostra aviso quando está próximo do limite
        showLoopWarning() {
            
            if (typeof Swal !== 'undefined') {
                Swal.fire({
                    title: 'Atenção',
                    text: 'Esta é sua última oportunidade de completar o processo.',
                    icon: 'warning',
                    timer: 4000,
                    showConfirmButton: false
                });
            }
        },
        
        // Redireciona para a URL de saída
        redirectToExit() {
            this.resetLoopData();
            window.location.href = this.config.exitUrl;
        },
        
        // Configura handlers para saída da página
        setupExitHandlers() {
            // Intercepta tentativas de fechar a aba/navegador
            window.addEventListener('beforeunload', (e) => {
                const loopCount = this.getLoopCount();
                if (loopCount >= this.config.maxLoops - 1) {
                    e.preventDefault();
                    e.returnValue = 'Tem certeza que deseja sair? Você está quase finalizando o processo.';
                    return e.returnValue;
                }
            });
        },
        
        // Força o redirecionamento para a próxima página (usado pelas páginas do funil)
        forceRedirectToNext(nextUrl) {
            const loopCount = this.getLoopCount();
            
            // Se atingiu o limite, redireciona para saída
            if (loopCount >= this.config.maxLoops) {
                this.redirectToExit();
                return false;
            }
            
            // Caso contrário, permite o redirecionamento normal
            window.location.href = nextUrl;
            return true;
        },
        
        // Obtém estatísticas do loop para debugging
        getStats() {
            return {
                loopCount: this.getLoopCount(),
                visitedPages: this.getVisitedPages(),
                startTime: this.getStartTime(),
                currentPage: this.getCurrentPage(),
                timeElapsed: this.getStartTime() ? Date.now() - this.getStartTime() : 0
            };
        }
    };
    
    // Expõe o LoopControl globalmente
    window.LoopControl = LoopControl;
    
    // Inicializa automaticamente quando o DOM estiver pronto
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => LoopControl.init());
    } else {
        LoopControl.init();
    }
    
    // Debug: expõe stats no console
    
})();